/* $Id: decomp.c,v 1.7 1998/07/20 21:05:58 ericb Exp $ */
/* decompress from C array */

/**************************************************************
    lzhuf.c
    written by Haruyasu Yoshizaki 11/20/1988
    some minor changes 4/6/1989
    comments translated by Haruhiko Okumura 4/7/1989
**************************************************************/

/* adapted from 3569 code, Don Hiller 1995-1997 */

#include "sema.h"

/********** LZSS compression **********/

#define N       4096    /* buffer size */
#define F       60  /* lookahead buffer size */
#define THRESHOLD   2


/* Huffman coding */

#define N_CHAR      (256 - THRESHOLD + F)
                /* kinds of characters (character code = 0..N_CHAR-1) */
#define T       (N_CHAR * 2 - 1)    /* size of table */
#define R       (T - 1)         /* position of root */
#define MAX_FREQ    0x8000      /* updates tree when the */
                    /* root frequency comes to this value. */

struct global_type
{
    unsigned long textsize;  /* total uncompressed length */

    /* Decode() state variables */
    int  i, j, k;
    int  r;
    unsigned long count;

    /* GetBit(), GetByte() state variables */
    unsigned int getbuf;
    int getlen8;  /* the former getlen - 8 */
    unsigned char *inBuf;

    char text_buf[N + F - 1];

    /* frequency table */
    unsigned int freq[T + 1];

    /* pointers to parent nodes, except for the */
    /* elements [T..T + N_CHAR - 1] which are used to get */
    /* the positions of leaves corresponding to the codes. */
    int prnt[T + N_CHAR];

    /* pointers to child nodes (son[], son[] + 1) */
    int son[T];
};


/* table for encoding and decoding the upper 6 bits of position */

/* for decoding */
static const char d_code[256] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
    0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
    0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02, 0x02,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
    0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09, 0x09,
    0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A, 0x0A,
    0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B, 0x0B,
    0x0C, 0x0C, 0x0C, 0x0C, 0x0D, 0x0D, 0x0D, 0x0D,
    0x0E, 0x0E, 0x0E, 0x0E, 0x0F, 0x0F, 0x0F, 0x0F,
    0x10, 0x10, 0x10, 0x10, 0x11, 0x11, 0x11, 0x11,
    0x12, 0x12, 0x12, 0x12, 0x13, 0x13, 0x13, 0x13,
    0x14, 0x14, 0x14, 0x14, 0x15, 0x15, 0x15, 0x15,
    0x16, 0x16, 0x16, 0x16, 0x17, 0x17, 0x17, 0x17,
    0x18, 0x18, 0x19, 0x19, 0x1A, 0x1A, 0x1B, 0x1B,
    0x1C, 0x1C, 0x1D, 0x1D, 0x1E, 0x1E, 0x1F, 0x1F,
    0x20, 0x20, 0x21, 0x21, 0x22, 0x22, 0x23, 0x23,
    0x24, 0x24, 0x25, 0x25, 0x26, 0x26, 0x27, 0x27,
    0x28, 0x28, 0x29, 0x29, 0x2A, 0x2A, 0x2B, 0x2B,
    0x2C, 0x2C, 0x2D, 0x2D, 0x2E, 0x2E, 0x2F, 0x2F,
    0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
    0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F,
};

static const char d_len[256] = {
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03, 0x03,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04, 0x04,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05, 0x05,
    0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
    0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
    0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
    0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
    0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
    0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06, 0x06,
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
    0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07, 0x07,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
    0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08,
};

/* formerly GetBit(), modified */
#define GETBIT(_x_)				\
    if (getlen8 <= 0) {				\
        getbuf |= *(inBuf++) << -getlen8;	\
        getlen8 += 7; /* 8-1 */			\
    } else {					\
        getlen8--;				\
    }						\
    if ( getbuf & 0x8000 ) _x_++;		\
    getbuf <<= 1;

/* formerly GetByte(), modified */
#define GETBYTE(_x_)				\
    if (getlen8 <= 0) {				\
        getbuf |= *(inBuf++) << -getlen8;	\
	/* getlen8 washes out */		\
    } else {					\
    	getlen8 -= 8;				\
    }						\
    _x_ = (getbuf >> 8) & 0xff;			\
    getbuf <<= 8;


/* initialization of tree */

static void StartHuff(struct global_type *gb)
{
    register int i, j;
    unsigned int *freq = gb->freq;
    int *son = gb->son;
    int *prnt = gb->prnt;

    for (i = 0; i < N_CHAR; i++) {
        freq[i] = 1;
        son[i] = i + T;
        prnt[i + T] = i;
    }
    i = 0; j = N_CHAR;
    while (j <= R) {
        freq[j] = freq[i] + freq[i + 1];
        son[j] = i;
        prnt[i] = prnt[i + 1] = j;
        i += 2; j++;
    }
    freq[T] = 0xffff;
    prnt[R] = 0;
}


/* reconstruction of tree */

static void reconst(struct global_type *gb)
{
    register int i, j, k;
    unsigned int f;
#ifndef	HPVXI_DOWNLOAD
    unsigned int l;
#else	/* HPVXI_DOWNLOAD */
    int h;
#endif	/* HPVXI_DOWNLOAD */
    unsigned int *freq = gb->freq;
    int *son = gb->son;
    int *prnt = gb->prnt;

    /* collect leaf nodes in the first half of the table */
    /* and replace the freq by (freq + 1) / 2. */
    j = 0;
    for (i = 0; i < T; i++) {
        if (son[i] >= T) {
            freq[j] = (freq[i] + 1) / 2;
            son[j] = son[i];
            j++;
        }
    }
    /* begin constructing tree by connecting sons */
    for (i = 0, j = N_CHAR; j < T; i += 2, j++) {
        k = i + 1;
        f = freq[j] = freq[i] + freq[k];
        for (k = j - 1; f < freq[k]; k--);
        k++;
#ifndef	HPVXI_DOWNLOAD
        l = (j - k) * sizeof(int);
        (void) memmove(&freq[k + 1], &freq[k], l);
        freq[k] = f;
        (void) memmove(&son[k + 1], &son[k], l);
        son[k] = i;
#else	/* HPVXI_DOWNLOAD - memmove not available */
	for ( h = j; h > k; h-- )
	{
	    freq[h] = freq[h-1];
	    son[h] = son[h-1];
	}
        freq[k] = f;
        son[k] = i;
#endif	/* HPVXI_DOWNLOAD */
    }
    /* connect prnt */
    for (i = 0; i < T; i++) {
        if ((k = son[i]) >= T) {
            prnt[k] = i;
        } else {
            prnt[k] = prnt[k + 1] = i;
        }
    }
}


/* increment frequency of given code by one, and update tree */

static void update(int c, struct global_type *gb)
{
    register unsigned int k;
    register int l;
    int i, j;
    unsigned int *freq = gb->freq;
    int *son = gb->son;
    int *prnt = gb->prnt;

    if (freq[R] == MAX_FREQ) {
        reconst(gb);
    }
    c = prnt[c + T];
    do {
        k = ++freq[c];

        /* if the order is disturbed, exchange nodes */
        if (k > freq[l = c + 1]) {
            while (k > freq[++l]);
            l--;
            freq[c] = freq[l];
            freq[l] = k;

            i = son[c];
            prnt[i] = l;
            if (i < T) prnt[i + 1] = l;

            j = son[l];
            son[l] = i;

            prnt[j] = c;
            if (j < T) prnt[j + 1] = c;
            son[c] = j;

            c = l;
        }
    } while ((c = prnt[c]) != 0);   /* repeat up to root */
}



static int DecodeChar(struct global_type *gb)
{
    register unsigned int c;
    int *son = gb->son;
    register int getlen8 = gb->getlen8;
    register unsigned int getbuf = gb->getbuf;
    register unsigned char *inBuf = gb->inBuf;

    c = son[R];

    /* travel from root to leaf, */
    /* choosing the smaller child node (son[]) if the read bit is 0, */
    /* the bigger (son[]+1} if 1 */
    while (c < T) {
        GETBIT(c)
        c = son[c];
    }
    c -= T;
    gb->getlen8 = getlen8;
    gb->getbuf = getbuf;
    gb->inBuf = inBuf;
    update(c, gb);
    return c;
}

static int DecodePosition(struct global_type *gb)
{
    register unsigned int i, j, c;
    register int getlen8 = gb->getlen8;
    register unsigned int getbuf = gb->getbuf;
    register unsigned char *inBuf = gb->inBuf;

    /* recover upper 6 bits from table */
    GETBYTE(i)
    c = (unsigned int)d_code[i] << 6;
    j = d_len[i];

    /* read lower 6 bits verbatim */
    j -= 2;
    while (j--) {
        i <<= 1;
        GETBIT(i)
    }
    gb->getlen8 = getlen8;
    gb->getbuf = getbuf;
    gb->inBuf = inBuf;
    return c | (i & 0x3f);
}


int i1432_decompress_init(char *inBuffer, int **gb_ptr, unsigned long textsize)
{

    struct global_type *gb;
    int i;

    if (textsize == 0)
        return 0;

    gb = (struct global_type *) malloc(sizeof(struct global_type));
    *gb_ptr = (int *) gb;

    if ( *gb_ptr )
    {
        gb->inBuf = (unsigned char *)inBuffer;
	/* initialize with 1 byte read into getbuf */
        gb->getbuf = *((gb->inBuf)++) << 8;
        gb->getlen8 = 0;

	gb->textsize = textsize;

        gb->count = 0;
        gb->r = N - F;

        gb->k = -1; /* no partial decode in progress */

        StartHuff(gb);

        for (i = 0; i < N - F; i++)
        {
            gb->text_buf[i] = ' ';
        }
    }

    return 0;
}

int i1432_decompress(char *outBuf, unsigned long outSize, int *gb_ptr)
{
    register int  i, k;
    int  j, r, c;
    register unsigned long count;
    struct global_type *gb;
    char *text_buf;
    unsigned long textsize;
    unsigned long targCount;


    gb = (struct global_type *) gb_ptr;

    text_buf = gb->text_buf;
    textsize = gb->textsize;
    count = gb->count;
    i = gb->i;
    j = gb->j;
    k = gb->k;
    r = gb->r;

    targCount = count + outSize;
    if ( targCount > textsize )
    {
        targCount = textsize;
    }

    /* finish up partial decoding from last pass, if needed */
    if ( k >= 0 )
    {
        while ( k < j && count < targCount ) {
            c = text_buf[(i + k) & (N - 1)];
            *(outBuf++) = (char) c;
            text_buf[r++] = c;
            r &= (N - 1);
            count++;
	    k++;
        }
        if ( k >= j ) k = -1; /* mark this inner loop as done */
    }

    while ( count < targCount ) {
        c = DecodeChar(gb);
        if (c < 256) {
            *(outBuf++) = (char) c;
            text_buf[r++] = c;
            r &= (N - 1);
            count++;
        } else {
            i = (r - DecodePosition(gb) - 1) & (N - 1);
            j = c - 255 + THRESHOLD;
            for (k = 0; k < j && count < targCount; k++) {
                c = text_buf[(i + k) & (N - 1)];
                *(outBuf++) = (char) c;
                text_buf[r++] = c;
                r &= (N - 1);
                count++;
            }
	    if ( k >= j ) k = -1; /* mark this inner loop as done */
        }
    }

    if ( count == textsize )
    {
        /* done, free gb */
	free(gb_ptr);
	return 1;
    }
    else
    {
	/* save state for next time */
        gb->count = count;
        gb->i = i;
        gb->j = j;
        gb->k = k;
        gb->r = r;
	return 0;
    }
}
